implementation module commondef


//	Version 1.0.1

//	Common types for the I/O system and their access-rules.


import	StdArray, StdBool, StdChar, StdClass, StdEnum, StdFunc, StdInt, StdList, StdMisc, StdReal, StdString
// PA: Rect type is defined in the ostypes module.
from	ostypes		import Rect
import	osrgn		// PA: required for IntersectRgnRect
/* RWS ---
import	events
from	mac_types	import	Rect, WindowPtr, Toolbox
from	pointer		import	LoadWord, StoreWord
from	quickdraw	import	QGetPort, QSetPort, GrafPtr, QLocalToGlobal, QGlobalToLocal,
							QNewRgn, QDisposeRgn, QRectRgn, QSectRgn, QEmptyRgn
*/
import	StdIOCommon


/* RWS ---
/*	Convert a KeyMap (returned by GetKeys) into Modifiers (5 Booleans of which altDown==False).
*/
KeyMapToModifiers :: !(!Int,!Int,!Int,!Int) -> Modifiers
KeyMapToModifiers (w1,word,w3,w4)
=	{shiftDown=shift<>0,optionDown=option<>0,commandDown=command<>0,controlDown=control<>0,altDown=False}
where
	shift	= word bitand ShiftMask
	option	= word bitand OptionMask
	command	= word bitand CommandMask
	control	= word bitand ControlMask

/*	Check the status of the keyboard yields (return/enter, return/enter still, command '.') down.
*/
KeyEventInfo :: !Int !Int !Int -> (Bool,Bool,Bool)
KeyEventInfo what message mods
|	what==KeyDownEvent	= (returnOrEnter, False, commandPeriod)
|	what==AutoKeyEvent	= (False, returnOrEnter, False)
						= (False, False, False)
where
	returnOrEnter		= ('\015'==key || '\003'==key) && ms+cmd==0
	commandPeriod		= '.'==key && ms==0 && cmd<>0
	ms					= (mods bitand 512)+(mods bitand 2048)+(mods bitand 4096)
	cmd					= mods bitand 256
	key					= toChar (message bitand 255)

/*	Conversion of modifiers as found in events.
*/
ModifiersToInt :: !Modifiers -> Int
ModifiersToInt {shiftDown,optionDown,commandDown,controlDown}
=	mask shiftDown 512 bitor (mask optionDown 2048 bitor (mask commandDown 256 bitor mask controlDown 4096))
where
	mask :: !Bool !Int -> Int
	mask down n	|	down	= n
							= 0

IntToModifiers :: !Int -> Modifiers
IntToModifiers flags
=	{	shiftDown	= FlagIsSet flags 512
	,	optionDown	= FlagIsSet flags 2048
	,	commandDown	= FlagIsSet flags 256
	,	controlDown	= FlagIsSet flags 4096
	,	altDown		= False
	}

FlagIsSet flags flag	:== (flags bitand flag) <> 0

ShiftMask				:== 1
OptionMask				:== 4
CommandMask				:== 32768
ControlMask				:== 8

/*	Conversion of (KeyDown/AutoKey/KeyUp) Event message field to ASCII and key code.
*/
getASCII :: !Int -> Char
getASCII message = toChar (message bitand 255)

getMacCode :: !Int -> Int
getMacCode message = (message>>8) bitand 255

/*	Conversion of (KeyDown/AutoKey/KeyUp) Event what field to KeyState.
*/
keyEventToKeyState :: !Int -> KeyState
keyEventToKeyState KeyDownEvent	= KeyDown False
keyEventToKeyState AutoKeyEvent	= KeyDown True
keyEventToKeyState KeyUpEvent	= KeyUp


/*	GrafPort access rules:
*/
InGrafport :: !WindowPtr !(St Toolbox .x) !Toolbox -> (!.x, !Toolbox)
InGrafport wPtr f tb
#	(port,tb)	= QGetPort tb
	tb			= QSetPort wPtr tb
	(x,tb)		= f tb
	tb			= QSetPort port tb
=	(x,tb)

InGrafport2 :: !WindowPtr !(IdFun Toolbox) !Toolbox -> Toolbox
InGrafport2 wPtr f tb
#	(port,tb)	= QGetPort tb
	tb			= QSetPort wPtr tb
	tb			= f tb
	tb			= QSetPort port tb
=	tb

LocalToGlobal :: !Point !Toolbox -> (!Point,!Toolbox)
LocalToGlobal {x,y} tb
#	(x,y,tb)	= QLocalToGlobal x y tb
=	({x=x,y=y},tb)

GlobalToLocal :: !Point !Toolbox -> (!Point,!Toolbox)
GlobalToLocal {x,y} tb
#	(x,y,tb)	= QGlobalToLocal x y tb
=	({x=x,y=y},tb)

/*	Mouse access functions:
*/
GetMousePosition :: !Toolbox -> (!Point, !Toolbox)
GetMousePosition tb
#	(x,y,tb)	= GetMouse tb
=	({x=x,y=y},tb)

WaitForMouseUp :: !Toolbox -> Toolbox
WaitForMouseUp tb
#	(mouseDown,tb)	= WaitMouseUp tb
|	mouseDown		= WaitForMouseUp tb
					= tb
--- RWS */

/*	Extensions of StdFunc:
*/
K` :: .x !.y -> .y
K` _ y = y


/*	Calculation rules on Integers:
*/
Dist :: !Int !Int -> Int
Dist x y
|	d>=0	= d
			= y-x
where
	d		= x-y

SetBetween :: !Int !Int !Int -> Int
SetBetween x low up
|	x<=low	= low
|	x>=up	= up
			= x

IsBetween :: !Int !Int !Int -> Bool
IsBetween x low up
|	x<low	= False
			= x<=up

minmax :: !Int !Int -> (!Int,!Int)
minmax a b
	| a<=b	= (a,b)
			= (b,a)


/*	Calculation rules on Points, Sizes, and Vectors:
*/
addPointVector :: !Vector !Point -> Point
addPointVector {vx,vy} {x,y} = {x=x+vx,y=y+vy}

addPointSize :: !Size !Point -> Point
addPointSize {w,h} {x,y} = {x=x+w,y=y+h}

/* RWS ---
/*	Rules on Rects and Rectangles:
*/
LoadRect :: !Ptr !Toolbox -> (!Rect,!Toolbox)
LoadRect ptr tb
#	(top,   tb)	= LoadWord ptr		tb
	(left,  tb)	= LoadWord (ptr+2)	tb
	(bottom,tb)	= LoadWord (ptr+4)	tb
	(right, tb)	= LoadWord (ptr+6)	tb
=	((left,top, right,bottom),tb)

StoreRect :: !Ptr !Rect !Toolbox -> Toolbox
StoreRect ptr (left,top, right,bottom) tb
#	tb			= StoreWord ptr		top		tb
	tb			= StoreWord (ptr+2)	left	tb
	tb			= StoreWord (ptr+4)	bottom	tb
	tb			= StoreWord (ptr+6)	right	tb
=	tb

PA: the following functions on Rect and Rectangle are exported because they are used for window/control handling. */

ZeroRect :: Rect
ZeroRect = (0,0, 0,0)

RectangleToRect :: !Rectangle -> Rect
RectangleToRect {corner1={x=a,y=b},corner2={x=a`,y=b`}}
|	x_less_x` && y_less_y`	= (a,b,a`,b`)
|	x_less_x`				= (a,b`,a`,b)
|	y_less_y`				= (a`,b,a,b`)
							= (a`,b`,a,b)
where
	x_less_x` = a<=a`
	y_less_y` = b<=b`

RectToRectangle :: !Rect -> Rectangle
RectToRectangle (l,t, r,b) = {corner1={x=l,y=t},corner2={x=r,y=b}}

IsEmptyRect :: !Rect -> Bool
IsEmptyRect (l,t, r,b)		= l==r || t==b

IsEmptyRectangle :: !Rectangle -> Bool
IsEmptyRectangle {corner1,corner2}	= corner1.x==corner2.x || corner1.y==corner2.y

PointInRect :: !Point !Rect -> Bool
PointInRect {x,y} (l,t, r,b) = IsBetween x l r && IsBetween y t b

PointInRectangle :: !Point !Rectangle -> Bool
PointInRectangle point rectangle = PointInRect point (RectangleToRect rectangle)

PosSizeToRect :: !Point !Size -> Rect
PosSizeToRect {x,y} {w,h} = (x,y, x+w,y+h)

PosSizeToRectangle :: !Point !Size -> Rectangle
PosSizeToRectangle pos=:{x,y} {w,h} = {corner1=pos,corner2={x=x+w,y=y+h}}

SizeToRect :: !Size -> Rect
SizeToRect {w,h} = (0,0, w,h)

SizeToRectangle :: !Size -> Rectangle
SizeToRectangle {w,h} = {corner1=zero,corner2={x=w,y=h}}

DisjointRects :: !Rect  !Rect -> Bool
DisjointRects rect1 rect2
=	IsEmptyRect rect1 || IsEmptyRect rect2 || l1>=r2 || b1<=t2 || r1<=l2 || t1>=b2
where
	(l1,t1, r1,b1)	= rect1
	(l2,t2, r2,b2)	= rect2

IntersectRects :: !Rect !Rect -> Rect
IntersectRects rect1 rect2
|	DisjointRects rect1 rect2
=	ZeroRect
=	(max l1 l2,max t1 t2, min r1 r2,min b1 b2)
where
	(l1,t1, r1,b1)	= rect1
	(l2,t2, r2,b2)	= rect2

AddRectVector :: !Vector !Rect -> Rect
AddRectVector {vx,vy} (l,t, r,b) = (l+vx,t+vy, r+vx,b+vy)

AddRectangleVector :: !Vector !Rectangle -> Rectangle
AddRectangleVector v {corner1,corner2} = {corner1=addPointVector v corner1,corner2=addPointVector v corner2}

RectSize :: !Rect -> Size
RectSize (l,t, r,b) = {w=abs (r-l),h=abs (b-t)}


/*	Rules on RgnHandles and Rects:
*/
IntersectRgnRect :: !OSRgnHandle !Rect !*OSToolbox -> (!OSRgnHandle,!*OSToolbox)
IntersectRgnRect rgnH rect tb
	# (aidRgn,tb)	= osnewrgn tb
	# (aidRgn,tb)	= osrectrgn rect aidRgn tb
	# (secRgn,tb)	= ossectrgn rgnH aidRgn tb
	# tb			= osdisposergn   aidRgn tb
	= (secRgn,tb)

/*	PA: DisjointRgnRect made invisible
DisjointRgnRect :: !RgnHandle !Rect !Toolbox -> (!Bool,!Toolbox)
DisjointRgnRect rgnH rect tb
#	(aidRgn, tb)	= IntersectRgnRect rgnH rect tb
	(isEmpty,tb)	= QEmptyRgn aidRgn tb
	tb				= QDisposeRgn aidRgn tb
=	(isEmpty,tb)
*/

/*	Conversion of Rectangle to UpdateState:
*/
RectangleToUpdateState :: !Rectangle -> UpdateState
RectangleToUpdateState frame
	= {oldFrame=frame,newFrame=frame,updArea=[frame]}

/*	PA: two new functions that are used frequently:
	Conversion of Size and Point to tuples:
*/
SizeToTuple :: !Size  -> (!Int,!Int)
SizeToTuple {w,h} = (w,h)

PointToTuple:: !Point -> (!Int,!Int)
PointToTuple {x,y} = (x,y)

VectorToTuple:: !Vector -> (!Int,!Int)
VectorToTuple {vx,vy} = (vx,vy)

/*	Error generation rule:
*/
Error :: !String !String !String -> .x
Error rule moduleName error
=	abort ("Error in rule "+++rule+++" ["+++moduleName+++"]: "+++error+++".\n")

//	PA: new version of Error to dump fatal errors.
FatalError :: !String !String !String -> .x
FatalError rule moduleName error
=	abort ("Fatal error in rule "+++rule+++" ["+++moduleName+++"]: "+++error+++".\n")


/*	Universal dummy value (!!evaluation causes termination with the message: "Fatal error: dummy evaluated!"!!)
*/
dummy :: String -> .x
dummy error = abort ("Fatal error: dummy evaluated! "+++error+++".\n")


/*	Max Integer constants:
*/
MaxSigned2ByteInt	:== 32767		// 2^15-1
MaxSigned4ByteInt	:== 2147483647	// 2^31-1


/*	Bound data type:
*/
::	Bound
	=	Finite Int												// Fix a finite positive bound of N
	|	Infinite												// No bound

instance == Bound
where
	(==) :: !Bound !Bound -> Bool
	(==) (Finite i)	(Finite j)	= i==j || i<=0 && j<=0
	(==) Infinite	Infinite	= True
	(==) _			_			= False

zeroBound:: !Bound -> Bool
zeroBound (Finite i)	= i<=0
zeroBound _				= False

decBound :: !Bound -> Bound
decBound bound=:(Finite i)
|	i<=0		= Finite 0
				= Finite (i-1)
decBound bound	= bound


/*	PA: code changed and moved to oswindow.
/*	Standard Scroll Bar settings:
	Internally, scrollbars always have the following internal range:
	*	if the SliderState is not empty (sliderMin<>sliderMax): (StdSliderMin,StdSliderMax)
	*	if the SliderState is empty     (sliderMin==sliderMax): (StdSliderMin,StdSliderMin).
	The thumb is always set proportionally (see toSliderRange).
	Its value can be recalculated by fromSliderRange given the actual SliderState range.
*/
StdSliderMin		:== 0			// 0
StdSliderMax		:== 32767		// MaxSigned2ByteInt
StdSliderRange		:== 32767		// StdSliderMax-StdSliderMin

fromSliderRange :: !Int !Int !Int -> Int
fromSliderRange min max x
	| min==max
	= min
	= toInt ((toReal (x-StdSliderMin))*k)+min
where
	k = toReal (max-min) / toReal StdSliderRange

toSliderRange :: !Int !Int !Int -> Int
toSliderRange min max x
	| min==max
	= StdSliderMin
	= toInt ((toReal (x-min))/k)+StdSliderMin
where
	k = toReal (max-min) / toReal StdSliderRange
*/


/*	List operations:
*/
::	Cond  x :== x -> Bool
::	UCond x :== x -> (Bool,x)

IsSingleton :: ![.x] -> Bool
IsSingleton [x]	= True
IsSingleton _	= False

HdTl :: ![.x] -> (.x, [.x])
HdTl [x:xs]		= (x,xs)

InitLast :: ![.x] -> ([.x],.x)
InitLast [x] = ([],x)
InitLast [x:xs]
	# (init,last)	= InitLast xs
	= ([x:init],last)

Split :: !Int ![.x] -> (![.x],![.x])
Split _ []
=	([],[])
Split n xs
|	n<=0
=	([],xs)
#	(x, xs)	= HdTl xs
	(ys,zs)	= Split (n-1) xs
=	([x:ys],zs)

CondMap :: (Cond x) !(IdFun x) ![x] -> (!Bool,![x])
CondMap c f [x:xs]
#	(b,xs)	= CondMap c f xs
|	c x	
=	(True,[f x:xs])
=	(b,	[x:xs])
CondMap _ _ _
=	(False, [])

Uspan :: !(UCond .a) ![.a] -> (![.a],![.a])		// Same as span (StdList), but preserving uniqueness
Uspan c [x:xs]
	# (keep,x) = c x
	| keep
	= ([x:ys],zs)
	with
		(ys,zs) = Uspan c xs
	= ([],[x:xs])
Uspan _ _
	= ([],[])

FilterMap :: !(.x -> (Bool,.y)) ![.x] -> [.y]
FilterMap f [x:xs]
#!	(keep,y)	= f x
	ys			= FilterMap f xs
|	keep
=	[y:ys]
=	ys
FilterMap _ _
=	[]

StateMap :: !(.x -> .s -> (.y,.s)) ![.x] !.s -> (![.y],!.s)
StateMap f [x:xs] s
#!	(y, s)	= f x s
	(ys,s)	= StateMap f xs s
=	([y:ys],s)
StateMap _ _ s
=	([],s)

StateMap2 :: (.x -> .s -> .s) ![.x] !.s -> .s
StateMap2 f [x:xs] s
=	StateMap2 f xs (f x s)
StateMap2 _ _ s
=	s

StrictSeq :: ![.(.s -> .s)] !.s	-> .s		// Same as seq (StdFunc), but with strict state argument
StrictSeq [f:fs] s
	= StrictSeq fs (f s)
StrictSeq _ s
	= s

StrictSeqList :: !.[.St .s .x] !.s -> (![.x],!.s)	// Same as seqList (StdFunc), but with strict state argument
StrictSeqList [f:fs] s
	# (x, s) = f s
	# (xs,s) = StrictSeqList fs s
	= ([x:xs],s)
StrictSeqList _ s
	= ([],s)

Contains :: (Cond x) ![x] -> Bool
Contains c [x:xs]
|	c x
=	True
=	Contains c xs
Contains _ _
=	False

UContains :: (UCond .x) ![.x] -> (!Bool,![.x])
UContains c [x:xs]
#	(cond,x) = c x
|	cond
=	(True,[x:xs])
#	(b,xs) = UContains c xs
=	(b,[x:xs])
UContains _ _
=	(False,[])

Select :: (Cond x) x ![x] -> (!Bool, x)
Select c n [x:xs]
|	c x
=	(True,x)
=	Select c n xs
Select _ n _
=	(False,n)

Access :: (St .x (Bool,.y)) .y ![.x] -> (!Bool,.y,![.x])
Access acc n [x:xs]
#	((cond,y),x) = acc x
|	cond
=	(True,y,[x:xs])
#	(b,y,xs) = Access acc n xs
=	(b,y,[x:xs])
Access _ n _
=	(False,n,[])

AccessList :: (St .x .y) ![.x] -> (![.y],![.x])
AccessList acc [x:xs]
#	(y, x)	= acc x
	(ys,xs)	= AccessList acc xs
=	([y:ys],[x:xs])
AccessList _ _
=	([],[])

Remove :: (Cond x) x ![x] -> (!Bool,x,![x])
Remove c n [x:xs]
|	c x
=	(True,x,xs)
#	(b,y,xs) = Remove c n xs
=	(b,y,[x:xs])
Remove _ n _
=	(False,n,[])

URemove :: (UCond .x) .x ![.x] -> (!Bool,.x,![.x])
URemove c n [x:xs]
#	(cond,x)	= c x
|	cond
=	(True,x,xs)
#	(b,y,xs)	= URemove c n xs
=	(b,y,[x:xs])
URemove _ n _
=	(False,n,[])

Replace :: (Cond x) x ![x] -> (!Bool,![x])
Replace c y [x:xs]
|	c x
=	(True,[y:xs])
#	(b,xs)	= Replace c y xs
=	(b,[x:xs])
Replace _ _ _
=	(False,[])

UReplace :: (UCond .x) .x ![.x] -> (!Bool,![.x])
UReplace c y [x:xs]
#	(cond,x)= c x
|	cond
=	(True,[y:xs])
#	(b,xs)	= UReplace c y xs
=	(b,[x:xs])
UReplace _ _ _
=	(False,[])

Ulength :: ![.x] -> (!Int,![.x])
Ulength [x:xs]
#	(length,xs)= Ulength xs
=	(length+1,[x:xs])
Ulength _
=	(0,[])

RemoveCheck :: x !u:[x] -> (!Bool, !u:[x])	| Eq x
RemoveCheck y [x:xs]
|	y==x
=	(True,xs)
#	(b,xs)	= RemoveCheck y xs
=	(b,[x:xs])
RemoveCheck _ _
=	(False,[])

RemoveSpecialChars :: ![Char] !{#Char} -> {#Char}
RemoveSpecialChars sc string
	= {c\\c<-RemoveSpecialChars` sc [c\\c<-:string]}
where
	RemoveSpecialChars` :: ![Char] ![Char] -> [Char]
	RemoveSpecialChars` sc [c1:cs1=:[c2:cs2]]
		| isMember c1 sc
		= [c2:RemoveSpecialChars` sc cs2]
		= [c1:RemoveSpecialChars` sc cs1]
	RemoveSpecialChars` sc [c]
		| isMember c sc
		= []
		= [c]
	RemoveSpecialChars` _ _
		= []

disjointLists :: ![x] ![x] -> Bool	| Eq x
disjointLists xs ys
|	isEmpty xs || isEmpty ys
=	True
|	shorter xs ys
=	disjointLists` xs ys
=	disjointLists` ys xs
where
	shorter :: ![x] ![x] -> Bool
	shorter [] _			= True
	shorter [x:xs] [y:ys]	= shorter xs ys
	shorter _ _				= False
	
	disjointLists` :: ![x] ![x] -> Bool	| Eq x
	disjointLists` [x:xs] ys
	=	not (isMember x ys) && disjointLists` xs ys
	disjointLists` _ _
	=	True

noDuplicates :: ![x] -> Bool	| Eq x
noDuplicates [x:xs]
=	not (isMember x xs) && noDuplicates xs
noDuplicates _
=	True
